/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#ifndef RL_TYPES_H_
#define RL_TYPES_H_

#include "rlAPI.h"

#include <stdbool.h>

#define rlSNMAXLEN 40
#define rlIPMAXLEN 60
#define rlTIMEZONEMAXLEN 128

#define rlSCANNERINFOMAXLEN 256
#define rlSCANNERINFODESCRIPTIONMAXLEN 2048

#define rlMETADATAMAXLEN 2048

#define rlMAXSCANHEADS 2
#define rlSPIModuleCount 4
#define rlQuadratureDecoderModuleCount 2

#define rlMAXONOFFPATTERNLEN 128

#define rlCORRECTIONHASHLEN 128

#define rlNEWLINEMAXLEN 128

#define rlADAPTERNAMEMAXLEN 256

typedef enum
{
	rlERROR = -1,
	rlSUCCESS = 0
} rlResult;

typedef int32_t rlHandle;
typedef int64_t rlFontHandle;
typedef int64_t rlCommandListHandle;
typedef int64_t rlCallbackHandle;
#define rlInvalidHandle rlERROR

typedef enum
{
	lsNotExisting = 0,
	lsIdle = 1,
	lsInExecutionQueue = 2,
	lsProcessing = 3
} rlListStatus;

typedef enum
{
	lsNone = 0,
	lsPilot = 1,
	lsArmed = 2,
	lsAlarm = 4,
} rlLaserStatus;

typedef enum
{
	plActiveHigh = 0,
	plActiveLow = 1
} rlPolarity;

typedef enum
{
	etListDone = 1 << 0,
	etListIdle = 1 << 1,
	etListIdleAtLimit = 1 << 21
} rlEventType;

typedef enum
{
	ioPortA = 0,
	ioPortB = 1,
	ioPortC = 2,
	ioPortD = 3,
	ioPortE = 4,
	ioLaserIn = 11,
	ioLaserOut = 12,
} rlIOPort;

typedef enum
{
	ioNone = 0,
	ioZero = 1,
	ioThreePointThree = 2,
	ioFive = 3
} rlIOVoltage;

typedef enum
{
	ioInput = 0,
	ioOutput = 1,
} rlIODirection;

typedef enum
{
	Sfr0 = 0,
	Sfr1 = 1,
} rlSfRegister;

typedef enum
{
	sfr0None = 0,
	sfr0StartMark = 1 << 0,
	sfr0AbortMark = 1 << 1,
	sfr0PartSensor = 1 << 2,
	sfr0LaserAlarm = 1 << 7,
	sfr0StartMarkAsserted = sfr0StartMark << 8,
	sfr0AbortMarkAsserted = sfr0AbortMark << 8,
	sfr0PartSensorAsserted = sfr0PartSensor << 8,
	sfr0LaserAlarmAsserted = sfr0LaserAlarm << 8,
	sfr0StartMarkDeasserted = sfr0StartMark << 16,
	sfr0AbortMarkDeasserted = sfr0AbortMark << 16,
	sfr0PartSensorDeasserted = sfr0PartSensor << 16,
	sfr0LaserAlarmDeasserted = sfr0LaserAlarm << 16,
} rlSfr0Bits;

typedef enum
{
	sfr1None = 0,
	sfr1QA0 = 0x0100,
	sfr1QB0 = 0x0200,
	sfr1Index0 = 0x0400,
	sfr1QA1 = 0x1000,
	sfr1QB1 = 0x2000,
	sfr1Index1 = 0x4000,
	sfr1PartSensor = 0x10000,
} rlSfr1Bits;

typedef enum
{
	dsBINFILES = 0,
	dsCORRFILES = 1,
	dsJOBFILES = 2,
	dsLISTFILES = 3,
	dsPROCVARFILES = 4
} rlDirectorySelector;

typedef enum
{
	ptNone = 0,
	ptDac0 = 1,
	ptDac1 = 2,
	ptDigital1Bit = 3,
	ptDigital2Bit = 4,
	ptDigital8Bit = 5,
	ptDigital16Bit = 6,
	ptLmFrequency = 11,
	ptLmWidth = 12
} rlPowerTarget;

typedef enum
{
	ptcNone = 0,
	ptcChannel0 = 1 << 0,
	ptcChannel1 = 1 << 1,
	ptcAllChannels = 0xFF
} rlPowerChannels;

typedef enum
{
	L1 = -1,
	L2 = 0,
	L3 = 1,
	L5 = 2,
	L9 = 3,
	L17 = 4,
	L33 = 5,
	L65 = 6,
	L129 = 7,
	L257 = 8
} rlLayers;

typedef enum
{
	aiX = 0,
	aiY = 1,
	aiZ = 2,
	aiZoomZ = 3,
	aiAuxiliary = 4,
	aiSensorZ = 5
} rlAxis;

typedef enum
{
	axNone = 0x00,
	axX = 0x01,
	axY = 0x02,
	axZ = 0x04,
	axZoomZ = 0x08,
	axAuxiliary = 0x10,
	axSensorZ = 0x20,
	axXY = axX | axY, // for convenience when working with 2D
	axXYZ = axX | axY | axZ
} rlAxes;

typedef enum
{
	XY2 = 0 << 0,
	SL2 = 1 << 0,
	RL3 = 2 << 0,
	Defocus = 1 << 2,
	SingleHead = 0 << 3,
	DualHead = 1 << 3,
	Auxiliary = 1 << 4,
	Extended = 1 << 5,
	FourD = 1 << 6
} rlScanHeadFlags;

typedef enum
{
	// shfXY2_100 = XY2 | SingleHead,		// Obsolete: Please use shfXY2_Single3D instead.
	shfXY2_Single3D = XY2 | SingleHead,
	shfXY2_Dual3D = XY2 | DualHead,

	shfSL2_Single3D = SL2 | SingleHead,
	shfSL2_Single3DX = SL2 | SingleHead | Extended,
	shfSL2_Single3DA = SL2 | SingleHead | Auxiliary,

	shfSL2_Single4DF = SL2 | SingleHead | FourD | Defocus,
	shfSL2_Single4DFX = SL2 | SingleHead | FourD | Defocus | Extended,
	shfSL2_Single4DFA = SL2 | SingleHead | FourD | Defocus | Auxiliary,

	shfSL2_Single4D = SL2 | SingleHead | FourD,

	shfSL2_Dual2D = SL2 | DualHead,

	shfRL3_Single3D = RL3 | SingleHead,
	shfRL3_Single3DX = RL3 | SingleHead | Extended,
	shfRL3_Single3DA = RL3 | SingleHead | Auxiliary,
	shfRL3_Single3DXA = RL3 | SingleHead | Extended | Auxiliary,

	shfRL3_Single4DF = RL3 | SingleHead | FourD | Defocus,
	shfRL3_Single4DFX = RL3 | SingleHead | FourD | Defocus | Extended,
	shfRL3_Single4DFA = RL3 | SingleHead | FourD | Defocus | Auxiliary,
	shfRL3_Single4DFXA = RL3 | SingleHead | FourD | Defocus | Extended | Auxiliary,

	shfRL3_Single4D = RL3 | SingleHead | FourD,
	shfRL3_Single4DX = RL3 | SingleHead | FourD | Extended,
	shfRL3_Single4DA = RL3 | SingleHead | FourD | Auxiliary,
	shfRL3_Single4DXA = RL3 | SingleHead | FourD | Extended | Auxiliary,

	shfRL3_Dual3D = RL3 | DualHead
} rlScanHeadFormat;

typedef enum
{
	paClear = 0,
	paToggle = 1,
	paSet = 2,
	paWrite = 3
} rlPinAction;

typedef enum
{
	setDataMode = 0x05000,
	updatePermanateMemory = 0x0A000,
	selectTuning = 0x11000,
	setPositionScaleFactor = 0x12000,
	setPosAcknowledgeLevel = 0x15000,
	dataModeStorage = 0x17000,
	setEchoMode = 0x21000
} rlEPCommandCode;

typedef enum
{
	statusWord = 0x00000,
	currentPosition = 0x00010,
	targetPosition = 0x00020,
	positionError = 0x00030,
	outputCurrent = 0x00040,
	relativeOutputCurrent = 0x00050,
	currentVelocity = 0x00060,
	galvoTemperature = 0x00140,
	servoBoardTemperature = 0x00150,
	pdAGCVoltage = 0x00160,
	dspCoreVoltage = 0x00170,
	dspIOVoltage = 0x00180,
	analogSupplyVoltage = 0x00190,
	adcSupplyVoltage = 0x001A0,
	pdAGCCurrent = 0x001B0,
	relativeHeatingOutput = 0x001D0,
	serialNumberLow = 0x001E0,
	serialNumberHigh = 0x001F0,
	articleNumberLow = 0x00200,
	articleNumberHigh = 0x00210,
	firmwareVersion = 0x00220,
	calibrationFactor = 0x00230,
	aperture = 0x00240,
	wavelength = 0x00250,
	tuningNumber = 0x00260,
	dataModeNumber = 0x00270,
	stateFlagsLow = 0x00280,
	stateFlagsHigh = 0x00290,
	stopEventCode = 0x002A0,
	stopFlagsLow = 0x002B0,
	stopFlagsHigh = 0x002C0,
	runningTimeSeconds = 0x002F0,
	runningTimeMinutes = 0x00300,
	runningTimeHours = 0x00310,
	runningTimeDays = 0x00320,
	positionScale = 0x003F0,
	raylaseStatusWord = 0x00800,
} rlEPDataMode;

typedef enum
{
	changeScaleFactor = 0x00830,

	scaleFactorOne = 0x00000,
	scaleFactorHalf = 0x00010,
	scaleFactorQuarter = 0x00020,
	scaleFactorEighth = 0x00030
} rlEPPositionScaleFactor;

typedef enum
{
	dmSave = 0x00FF0,
	dmRecall = 0x00000,
} rlEPDMStorage;

typedef enum
{
	smSyncPerWord = 0,
	smSyncPerFrame = 1,
	smMarkEndOfWord = 2,
	smMarkEndOfFrame = 3,
	smMarkPostWord = 4
} rlSyncMode;

typedef enum
{
	deSpi = 0,
	deSpiListCommand = 1,
	deScannerCommunication = 2,
	deFrameSync = 3
} rlDataSource;

typedef enum
{
	stInactive = 0,
	stActive = 1
} rlState;

typedef enum
{
	orMsbFirst = 0,
	orLsbFirst = 1
} rlOrder;

typedef enum
{
	omStandard = 0,
	omContinuous = 1
} rlMotfOperatingMode;

typedef enum
{
	dmRelativeToCurrent = 0,
	dmRelativeToPrevious = 1
} rlMotfDistanceMode;

typedef enum
{
	stQuadratureDecoder = 0,
	stSpi = 1
} rlMotfSourceType;

typedef enum
{
	pyNone = 0,
	pyOdd = 1,
	pyEven = 2,
	pyMark = 3,
	pySpace = 4
} rlParity;

typedef enum
{
	hsNone = 0,
	hsXOnXOff = 1,
	hsRequestToSend = 2,
	hsRequestToSendXOnXOff = 3
} rlHandshake;

typedef enum
{
	sbNone = 0,
	sbOne = 1,
	sbTwo = 2,
	sbOnePointFive = 3
} rlStopBits;

typedef enum
{
	opsNormal = 0,
	opsConfigurationFailed = (1 << 0),
	opsInitializationFailed = (1 << 1),
	opsScannerError = (1 << 2)
} rlOperatingState;

typedef enum
{
	cdInternal = 0,
	cdExternalDHCP = 1,
	cdExternalP2P = 2
} rlCardDisposition;

typedef enum
{
	swmDisabled = 0,
	swmForceAtStartFinish = 1,
	swmMinimumCoH = 2
} rlSkyWritingMode;

typedef struct
{
	double X;
	double Y;
} rlPoint2D;

typedef struct
{
	double X;
	double Y;
	double Z;
} rlPoint3D;

typedef struct
{
	double X;
	double Y;
	double Z;
	double M;
} rlPoint4D;

typedef struct
{
	int32_t X;
	int32_t Y;
} rlPoint2DInt;

typedef struct
{
	int32_t X;
	int32_t Y;
	int32_t Z;
} rlPoint3DInt;

typedef struct
{
	int32_t X;
	int32_t Y;
	int32_t Z;
	int32_t M;
} rlPoint4DInt;

typedef struct
{
	rlPoint2D Position;
	double Delay;
	double ExposureTime;
} rlJumpAndShootItem2D;
typedef struct
{
	rlPoint3D Position;
	double Delay;
	double ExposureTime;
} rlJumpAndShootItem3D;

typedef struct
{
	rlIOPort Port;
	rlIODirection Direction;
	int32_t PinIndex;
	char Functions[4][60];
} rlIOPinMapping;

typedef struct
{
	int ConfigVersion;

	rlIOVoltage IOLevel;
	rlPolarity Polarities[32];
	uint32_t PolaritiesLen;
	rlIODirection Directions[32];
	uint32_t DirectionsLen;
	int32_t Functions[32];
	uint32_t FunctionsLen;
	double WriteDelay;
	double WriteWidth;
} rlPortConfig;

typedef struct
{
	rlIOPort Port;
	rlPortConfig Config;
} rlPortsDict;

typedef struct
{
	int ConfigVersion;

	rlPortsDict Ports[16];
	int32_t PortsLen;
} rlGpioConfig;

typedef struct
{
	bool EnableGlobally;
	int32_t Exclusions[64];
	int32_t ExclusionsLen;
} rlEventStatus;

typedef struct
{
	rlEventType type;
	rlEventStatus EventStatus;
} rlEventsDict;

typedef struct
{
	int ConfigVersion;

	rlEventsDict Events[16];
	int32_t EventsLen;
} rlEventConfig;

typedef struct
{
	double A[3][3];
	double b[3];
} rlTransform3D;

typedef struct
{
	int ConfigVersion;
	rlPolarity FpsPolarity;
	rlPolarity GatePolarity;
	rlPolarity LMPolarity;
	double FpsStart;
	double FpsWidth;
	double GateSetup;
	double GateHold;
	double LaserTriggerDelay;
	double PowerScale;
	double PowerScale1;
	double PowerChangeSetup;
	double PowerWriteDelay;
	double PowerWriteWidth;
	rlPowerTarget HotPowerTarget;
	rlPowerTarget HotPowerTarget1;
	rlPowerTarget SimmerPowerTarget;
	uint16_t SimmerPower;
	bool EnableTickle;
	double TickleFrequency;
	double TickleWidth;
	bool EnableLaserSync;
	bool EnableGateModulatesLm;
	bool EnablePowerCorrection;
	bool EnablePowerCorrection1;
	bool EnableVelocityCorrection;
	bool EnableVelocityCorrection1;
	bool EnablePowerCalibration;
	bool EnablePowerCalibration1;
	bool EnableEquallySpacedPulses;
} rlLaserConfig;

typedef struct
{
	double AccelerationTime;
	double TrackingError;
	double HeadBias;
} rlAxisParameterSet;

typedef struct
{
	int ConfigVersion;
	rlTransform3D FieldTransform;
	rlAxisParameterSet Axes[5];
	bool EnableFieldCorrection;
} rlScanHeadConfig;

typedef struct
{
	int ConfigVersion;
	rlPoint3D FieldSize;
	int32_t HeadCount;
	rlScanHeadConfig Heads[rlMAXSCANHEADS];
	double ScanHeadDelay;
	rlScanHeadFormat HeadFormat;
	double MaxMagnification;
} rlScannerConfig;

typedef struct
{
	int ConfigVersion;
	rlMotfSourceType SourceType;
	int32_t ModuleID;
} rlMotfSource;

typedef struct
{
	int ConfigVersion;
	rlMotfOperatingMode OperatingMode;
	rlMotfSource MainSource;
	bool UseCompensation;
	rlMotfSource CompensationSource;
	bool EnableDebounce;
	double DebounceTime;
	bool EnableSuppression;
	double SuppressionDistance;
	double PartDistance;
	double ResponseTime;
	rlPoint3D MainDirection;
	rlPoint3D CompensationDirection;
	rlPoint3D Offset;
} rlMotfConfig;

typedef struct
{
	rlSkyWritingMode Mode;
	double MinCoH;
	double MergedExtensionsMaxCoH;
	double AccelerationDelay;
	double DecelerationDelay;
	double ExtensionTime;
	double LaserOnDelay;
	double LaserOffDelay;
} rlSkyWritingSettings;

typedef struct
{
	int ConfigVersion;
	rlPoint2D Amplitude;
	rlPoint2D Frequency;
	rlPoint2D Phase;
} rlWobbleSettings;

typedef struct
{
	int ConfigVersion;
	double SpeedScale;
	double JumpSpeed;
	double JumpDelay;
	bool EnableVariableJumpDelay;
	double MinJumpDelay;
	double JumpLengthLimit;
	double MarkSpeed;
	double MarkDelay;
	double MagnificationSpeed;
	bool EnableVariablePolyDelay;
	double PolyDelay;
	double LaserOnDelay;
	double LaserOffDelay;
	double MaxDwellTime;
	double LmFrequency;
	double LmWidth;
	uint16_t Power;
	uint16_t Power1;
	rlTransform3D Transformation;
	rlPoint3D HeadingScale;
	double TileOffset;
	uint32_t OnOffPatternCount;
	double OnOffPattern[rlMAXONOFFPATTERNLEN];
	bool EnableDashedLine;
	double EvaluationLeadTime;

	rlSkyWritingSettings SkyWriting;
	rlWobbleSettings Wobble;
} rlProcessVariables;

typedef struct
{
	int ConfigVersion;
	int MinorVersion;
	rlAxes UsedAxes;
	rlScanHeadFormat HeadFormat;
	rlLayers XLayers;
	rlLayers YLayers;
	rlLayers ZLayers;
	rlLayers MLayers;
	double MaxAngleX;
	double MaxAngleY;
	double MaxMagnification;
	double Y2Z0;
	double X2Y;
	double O2X;
	double FocalLength;
	double MaxLensTravel;
	rlPoint3D FieldSize;
	rlPoint3D FieldOffset;
	double Aperture;
	double WorkingDistance;
	double Wavelength;
	double BitPerMicronZ0;
	double BitPerMicronZ1;
	double BitPerMicronE0;
	double BitPerMicronE1;
	uint32_t HashLen;
	char Hash[rlCORRECTIONHASHLEN];
	rlPoint4D InputScale;
	rlPoint4D InputOffset;
} rlOpticalParameters;

typedef struct
{
	char Model[rlSCANNERINFOMAXLEN];
	char IdentificationNumber[rlSCANNERINFOMAXLEN];
	char TechnicalDescription[rlSCANNERINFODESCRIPTIONMAXLEN];
	char Date[rlSCANNERINFOMAXLEN];
	char Version[rlSCANNERINFOMAXLEN];
} rlScannerInfo;

typedef struct
{
	int ConfigVersion;
	bool Enabled;
	rlSyncMode SpiSyncMode;
	uint16_t BitsPerWord;
	double PreDelay;
	double PostDelay;
	double FrameDelay;
	rlDataSource OutputSource;
	rlOrder BitOrder;
	double ClockPeriod;
} rlSpiModule;

typedef struct
{
	int ConfigVersion;
	rlSpiModule Modules[rlSPIModuleCount];
} rlSpiConfig;

typedef struct
{
	int ConfigVersion;
	bool SimulateEncoder;
	double SimulationSpeed;
	double SimulationAcceleration;
	bool InvertDirection;
	bool EnableIndex;
	double DistancePerCount;
} rlQuadratureDecoderModule;

typedef struct
{
	int ConfigVersion;
	rlQuadratureDecoderModule Modules[rlQuadratureDecoderModuleCount];
} rlQuadratureDecoderConfig;

typedef struct
{
	rlPoint4D NominalBeforeTransformation;
	rlPoint4D NominalAfterTransformation;
	rlPoint3D Laser;
	rlPoint3D User;
	rlPoint3D Motf;
	rlPoint4D Actual[rlMAXSCANHEADS];
	rlPoint3D Head[rlMAXSCANHEADS];
} rlPositions;

typedef struct
{
	uint32_t IdLen;
	char Id[rlTIMEZONEMAXLEN];
} rlTimeZone;

typedef struct
{
	int ConfigVersion;
	bool Enabled;
	bool UsePingPong;
	uint32_t Command;
	uint32_t Mask;
	uint32_t GoodValue;
	uint32_t Command1;
	uint32_t Mask1;
	uint32_t GoodValue1;
	int32_t Period;
	int32_t MinConsecutiveErrors;
	bool ContinueOnError;
	rlAxes ScanHeadAxes[rlMAXSCANHEADS];
} rlScannerMonitoringConfig;

typedef struct
{
	int ConfigVersion;
	rlTimeZone TimeZone;
	bool EnableStandAloneMode;
	int32_t StandAloneListID;
	int32_t StandAloneExecuteOnErrorListID;
	rlScannerMonitoringConfig ScannerMonitoring;
} rlSystemConfig;

typedef struct
{
	char SerialNumber[rlSNMAXLEN];
	int32_t Port;
	char Ip[rlIPMAXLEN];
} rlCardInfo;

typedef struct
{
	int ConfigVersion;
	bool Enabled;
	int BaudRate;
	rlParity Parity;
	int DataBits;
	rlStopBits StopBits;
	rlHandshake Handshake;
	int ReadTimeout;
	int WriteTimeout;
	uint32_t NewLineLen;
	char NewLine[rlNEWLINEMAXLEN];
} rlSerialPortConfig;

typedef struct
{
	int32_t MacroVectors;
	int32_t MicroVectors;
	double ComputationTime;
	double ExecutionTime;
	double ComputationTimePerMicroVector;
} rlListExecutionStats;

typedef struct
{
	char Key[rlMETADATAMAXLEN];
	char Value[rlMETADATAMAXLEN];
} rlCFMetadataItem;

typedef struct
{
	int32_t ConfigVersion;
	uint32_t CardID;
	uint32_t VendorID;
	uint32_t BoardVersion;
	uint32_t FpgaVersion;
	char Name[rlADAPTERNAMEMAXLEN];
	rlGpioConfig PortConfigCollection;
} rlAdapterBoardConfig;

typedef void (*rlListDoneCallback)(rlHandle, int32_t);
typedef void (*rlListIdleCallback)(rlHandle, int32_t);
typedef void (*rlProgressCallback)(rlHandle, int32_t);
typedef void (*rlHomeDoneCallback)(rlHandle, int32_t);
typedef void (*rlMoveDoneCallback)(rlHandle, int32_t);
typedef void (*rlErrorCallback)(rlHandle, int32_t, const char*, uint32_t);

#endif // RL_TYPES_H_
